package gov.va.med.mhv.vitals.validator;

import java.math.BigDecimal;
import java.util.Date;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import gov.va.med.mhv.common.api.util.ResponseUtil;
import gov.va.med.mhv.vitals.dto.LipidsReadingDTO;
import gov.va.med.mhv.vitals.enums.FastNonFastEnumeration;
import gov.va.med.mhv.vitals.util.MessagesUtil;

@Component
public class LipidsValidator extends Validator {
	
	public static final int COMMENTS_MAX_LENGTH = 255;
	private static final BigDecimal TOTAL_MIN_VALUE = new BigDecimal("10.0");
	private static final BigDecimal TOTAL_MAX_VALUE = new BigDecimal("2000.0");
	private static final BigDecimal LDL_MIN_VALUE = new BigDecimal("10.0");
	private static final BigDecimal LDL_MAX_VALUE = new BigDecimal("1000.0");
	private static final BigDecimal HDL_MIN_VALUE = new BigDecimal("6.0");
	private static final BigDecimal HDL_MAX_VALUE = new BigDecimal("600.0");
	public static final int PROVIDER_MAX_LENGTH = 100;
	private static final BigDecimal TRIGLYCERIDES_MIN_VALUE = new BigDecimal("10.0");
	private static final BigDecimal TRIGLYCERIDES_MAX_VALUE = new BigDecimal("2000.0");
	public static final int LOCATION_MAX_LENGTH = 100;
	public static final String FASTING =  "F";
	public static final String FASTING_BLANK = "B";


	
	@Autowired
	private MessagesUtil messages;
	
	public void lipidValidations(ResponseUtil response, LipidsReadingDTO lipiddto) {
		fieldValidation(response, lipiddto);
		if (response.getValidationErrors().size() == 0) {
			complexValidation(response, lipiddto);
		}

	}

	private void complexValidation(ResponseUtil response,LipidsReadingDTO lipiddto) {
		
		if(lipiddto.getFast().equals(FASTING)){
			if(lipiddto.getHdl()== null || lipiddto.getLdl() == null || lipiddto.getTriglycerides()==null){
				response.getValidationErrors().put("sei.lipids.detail.error.noEntries.fastingStatus.fasting",
						messages.getInvalidFasting());
			}
		}
	}

	private void fieldValidation(ResponseUtil response,	LipidsReadingDTO lipiddto) {
		validateComments(response,lipiddto);
		validateTotal(response,lipiddto);
		validateFastOrNonFast(response,lipiddto);
		validateLdl(response,lipiddto);
		validateDateEntered(response,lipiddto);
		validateHdl(response,lipiddto);
		validateProvider(response,lipiddto);
		validateTriglycerides(response,lipiddto);
		validateLocation(response,lipiddto);

		
	}

	private void validateLocation(ResponseUtil response,LipidsReadingDTO lipiddto) {
		String value = lipiddto.getLocation();
		if ( value == null ) {
			return;
		}						
		Validations.validateMaxLength("location",value, LOCATION_MAX_LENGTH, response.getValidationErrors());
	}

	private void validateTriglycerides(ResponseUtil response,LipidsReadingDTO lipiddto) {
		BigDecimal value = lipiddto.getTriglycerides();
		if( value == null ) {
			return;
		}		
		Validations.validateMinValue("triglycerides minumum", value, TRIGLYCERIDES_MIN_VALUE,"The Triglycerides ", response.getValidationErrors());
		Validations.validateMaxValue("triglycerides maximum",value, TRIGLYCERIDES_MAX_VALUE,"The Triglycerides ", response.getValidationErrors());
	}

	private void validateProvider(ResponseUtil response, LipidsReadingDTO lipiddto) {
		String value = lipiddto.getProvider();
		if ( value == null ) {
			return;
		}						
		Validations.validateMaxLength("provider",value, PROVIDER_MAX_LENGTH, response.getValidationErrors());
	}

	private void validateHdl(ResponseUtil response, LipidsReadingDTO lipiddto) {
		BigDecimal value = lipiddto.getHdl();
		if( value == null ) {
			return;
		}		
		Validations.validateMinValue("hdl minumum", value, HDL_MIN_VALUE,"The HDL ", response.getValidationErrors());
		Validations.validateMaxValue("hdl maximum",value, HDL_MAX_VALUE,"The HDL ", response.getValidationErrors());
	}

	private void validateDateEntered(ResponseUtil response,	LipidsReadingDTO lipiddto) {
		Date value = lipiddto.getReading();
		if( value == null ) {
			Validations.validateRequired("readingDate", value, response.getValidationErrors());
			return;
		}
		
	}

	private void validateLdl(ResponseUtil response, LipidsReadingDTO lipiddto) {
		BigDecimal value = lipiddto.getLdl();
		if( value == null ) {
			return;
		}		
		Validations.validateMinValue("ldl minumum", value, LDL_MIN_VALUE,"The LDL ", response.getValidationErrors());
		Validations.validateMaxValue("ldl maximum",value, LDL_MAX_VALUE,"The LDL ", response.getValidationErrors());
	}

	private void validateFastOrNonFast(ResponseUtil response,LipidsReadingDTO lipiddto) {
		String value=lipiddto.getFast();
		if ( value == null ) {
			Validations.validateRequired("fastOrNonFast", value, response.getValidationErrors());
			return;
		}	
		Validations.validateEnumeration("fastOrNonFast", value, FastNonFastEnumeration.values(), response.getValidationErrors());
	}

	private void validateTotal(ResponseUtil response, LipidsReadingDTO lipiddto) {
		BigDecimal value = lipiddto.getTotal();
		if( value == null ) {
			Validations.validateRequired("total", value, response.getValidationErrors());
			return;
		}		
		Validations.validateMinValue("total minumum", value, TOTAL_MIN_VALUE,"The Total ", response.getValidationErrors());
		Validations.validateMaxValue("total maximum",value, TOTAL_MAX_VALUE,"The Total ", response.getValidationErrors());
	}

	private void validateComments(ResponseUtil response,LipidsReadingDTO lipiddto) {
		String value = lipiddto.getComments();
		if ( value == null ) {
			return;
		}						
		Validations.validateMaxLength("comments",value, COMMENTS_MAX_LENGTH, response.getValidationErrors());
	}

}
